;ISN Shared Addons.au3
;Funcs in this file are used from the ISN Studio itself, as well as the helper thread

; #FUNCTION# ====================================================================================================================
; Name ..........: _ISN_RGB_To_ARGB
; Description ...: Converts an RGB Hex value to an Alpha-RGB value. The Alpha in this function is not adjustable and fixed to "FF".
; Syntax ........: _ISN_RGB_To_ARGB([$HexColor=0x000000])
; Parameters ....: $HexColor            - [optional] A handle value. Default is 0x000000.
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _ISN_RGB_To_ARGB($HexColor = 0x000000)
	Local $Blue = BitAND($HexColor, 0xFF)
	Local $Green = BitAND(BitShift($HexColor, 8), 0xFF)
	Local $Red = BitAND(BitShift($HexColor, 16), 0xFF)
	Local $color_argb = 0xFF000000 + $Red * 0x10000 + $Green * 0x100 + $Blue
	Return $color_argb
EndFunc   ;==>_ISN_RGB_To_ARGB

; #FUNCTION# ====================================================================================================================
; Name ..........: _RGB_to_BGR
; Description ...: Converts a Hex RGB value to an BGR format.
; Syntax ........: _RGB_to_BGR([$colour = ""])
; Parameters ....: $colour              - [optional] An unknown value. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _RGB_to_BGR($colour = "")
	If $colour = "" Then Return 0
	$r = _ColorGetRed($colour)
	$g = _ColorGetGreen($colour)
	$b = _ColorGetBlue($colour)
	$BGR = "0x" & Hex($b, 2) & Hex($g, 2) & Hex($r, 2)
	Return $BGR
EndFunc   ;==>_RGB_to_BGR

; #FUNCTION# ====================================================================================================================
; Name ..........: _BGR_to_RGB
; Description ...: Converts a Hex BGR value to an RGB format.
; Syntax ........: _BGR_to_RGB([$colour = ""])
; Parameters ....: $colour              - [optional] An unknown value. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _BGR_to_RGB($colour = "")
	If $colour = "" Then Return 0
	$r = _ColorGetBlue($colour)
	$g = _ColorGetGreen($colour)
	$b = _ColorGetRed($colour)
	$RGB = "0x" & Hex($r, 2) & Hex($g, 2) & Hex($b, 2)
	Return $RGB
EndFunc   ;==>_BGR_to_RGB

; #FUNCTION# ====================================================================================================================
; Name ..........: _ISN_GUICreateTopBackground
; Description ...: Creates the "blue" background header in the top of the GUI.
; Syntax ........: _ISN_GUICreateTopBackground([$GUIHandle="" [, $HeightOverride=-1 [, $WidthOverride=-1]]])
; Parameters ....: $GUIHandle           - [optional] An unknown value. Default is "".
;                  $HeightOverride      - [optional] An AutoIt controlID. Default is -1.
;                  $WidthOverride       - [optional] An AutoIt controlID. Default is -1.
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: This func also removes the CS_HREDRAW and CS_VREDRAW styles from the GUI to prevent flickering.
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _ISN_GUICreateTopBackground($GUIHandle = "", $HeightOverride = -1, $WidthOverride = -1)
	GUISwitch($GUIHandle)
	$GUI_Size = _WinGetClientSize($GUIHandle)
	If Not IsArray($GUI_Size) Then Return

	_WinAPI_SetClassLongEx($GUIHandle, $GCL_STYLE, BitAND(_WinAPI_GetClassLongEx($GUIHandle, $GCL_STYLE), BitNOT(1), BitNOT(2))) ;unset CS_HREDRAW and CS_VREDRAW flags. Prevent flickering on resite. thx to binhnx!

	Local $ImageX = 0
	Local $ImageY = 0
	Local $ImageWidth = $GUI_Size[0]
	Local $ImageHeight = 50
	Local $ThemeToUse = "light"
	If $ISN_Dark_Mode = "true" Or $WindowBackgroundIsDark Then $ThemeToUse = "dark" ;if used window color is dark

	If $WidthOverride <> -1 Then $ImageWidth = $WidthOverride * $DPI
	If $HeightOverride <> -1 Then $ImageHeight = $HeightOverride * $DPI

	Local $PicHandle = GUICtrlCreatePic("", $ImageX, $ImageY, $ImageWidth, $ImageHeight)
	If $PicHandle = 0 Then Return
	GUICtrlSetState($PicHandle, $GUI_DISABLE)
	GUICtrlSetResizing($PicHandle, 550)
	If $ThemeToUse = "light" Then
		_SetImage($PicHandle, @ScriptDir & "\Data\wintop.png", 0, True)
	Else
		_SetImage($PicHandle, @ScriptDir & "\Data\wintop_dark.png", 0, True)
	EndIf

	Return $PicHandle
EndFunc   ;==>_ISN_GUICreateTopBackground



; #FUNCTION# ====================================================================================================================
; Name ..........: _ISN_GUICreateRowBottomBackground
; Description ...: Creates the "blue" row at the bottom of the GUI.
; Syntax ........: _ISN_GUICreateRowBottomBackground([$GUIHandle="" [, $HeightOverride=-1 [, $WidthOverride=-1]]])
; Parameters ....: $GUIHandle           - [optional] An unknown value. Default is "".
;                  $HeightOverride      - [optional] An AutoIt controlID. Default is -1.
;                  $WidthOverride       - [optional] An AutoIt controlID. Default is -1.
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _ISN_GUICreateRowBottomBackground($GUIHandle = "", $HeightOverride = -1, $WidthOverride = -1)
	GUISwitch($GUIHandle)
	$GUI_Size = _WinGetClientSize($GUIHandle)
	If Not IsArray($GUI_Size) Then Return

	Local $ImageX = 0
	Local $ImageY = 0
	Local $ImageWidth = $GUI_Size[0]
	Local $ImageHeight = 50
	Local $ThemeToUse = "light"
	If $ISN_Dark_Mode = "true" Then $ThemeToUse = "dark"
	If $WindowBackgroundIsDark Then Return ;Overwrite, if used window color is dark

	If $WidthOverride <> -1 Then $ImageWidth = $WidthOverride * $DPI
	If $HeightOverride <> -1 Then $ImageHeight = $HeightOverride * $DPI
	$ImageY = $GUI_Size[1] - $ImageHeight
	If $ISN_Dark_Mode = "true" Then $ImageY = $ImageY + 3 * $DPI


	Local $PicHandle = GUICtrlCreatePic("", $ImageX, $ImageY, $ImageWidth, $ImageHeight)
	If $PicHandle = 0 Then Return
	GUICtrlSetState($PicHandle, $GUI_DISABLE)
	GUICtrlSetResizing($PicHandle, 582)
	If $ThemeToUse = "light" Then
		_SetImage($PicHandle, @ScriptDir & "\Data\row_bottom.png", 0, True)
	Else
		_SetImage($PicHandle, @ScriptDir & "\Data\row_bottom_dark.png", 0, True)
	EndIf

	Return $PicHandle
EndFunc   ;==>_ISN_GUICreateRowBottomBackground

; #FUNCTION# ====================================================================================================================
; Name ..........: _ISN_GUICreateBottomBackground
; Description ...: Creates the "blue" background at the bottom of the GUI.
; Syntax ........: _ISN_GUICreateBottomBackground([$GUIHandle="" [, $HeightOverride=-1 [, $WidthOverride=-1]]])
; Parameters ....: $GUIHandle           - [optional] An unknown value. Default is "".
;                  $HeightOverride      - [optional] An AutoIt controlID. Default is -1.
;                  $WidthOverride       - [optional] An AutoIt controlID. Default is -1.
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _ISN_GUICreateBottomBackground($GUIHandle = "", $HeightOverride = -1, $WidthOverride = -1)
	GUISwitch($GUIHandle)
	$GUI_Size = _WinGetClientSize($GUIHandle)
	If Not IsArray($GUI_Size) Then Return

	Local $ImageX = 0
	Local $ImageY = 0
	Local $ImageWidth = $GUI_Size[0]
	Local $ImageHeight = 45
	Local $ThemeToUse = "light"
	If $ISN_Dark_Mode = "true" Or $WindowBackgroundIsDark Then $ThemeToUse = "dark" ;if used window color is dark

	If $WidthOverride <> -1 Then $ImageWidth = $WidthOverride * $DPI
	If $HeightOverride <> -1 Then $ImageHeight = $HeightOverride * $DPI
	$ImageY = $GUI_Size[1] - $ImageHeight
	If $ISN_Dark_Mode = "true" Then $ImageY = $ImageY + 3 * $DPI

	Local $PicHandle = GUICtrlCreatePic("", $ImageX, $ImageY + 2, $ImageWidth, $ImageHeight) ;+2 at y is offset
	If $PicHandle = 0 Then Return
	GUICtrlSetState($PicHandle, $GUI_DISABLE)
	GUICtrlSetResizing($PicHandle, 582)
	If $ThemeToUse = "light" Then
		_SetImage($PicHandle, @ScriptDir & "\Data\winbottom.png", 0, True)
	Else
		_SetImage($PicHandle, @ScriptDir & "\Data\winbottom_dark.png", 0, True)
	EndIf

	Return $PicHandle
EndFunc   ;==>_ISN_GUICreateBottomBackground

; #FUNCTION# ====================================================================================================================
; Name ..........: _ISN_InputAddClearButton
; Description ...: Adds a grey "clear" button to an input control at the right corner of the control.
; Syntax ........: _ISN_InputAddClearButton([$InputHandle = "" [, $ResizeValue = 0 [, $ClickFunc = "" [, $BackgroundOverwrite = -1]]]])
; Parameters ....: $InputHandle         - [optional] Handle to the input control, where the clear button needs to be added.
;                  $ResizeValue         - [optional] Resize value to use for the icon.
;                  $ClickFunc           - [optional] Func to call when the user clicked on the clear icon.
;                  $BackgroundOverwrite - [optional] Define a fixed background color for the clear icon.
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......: NOTE: INPUT CONTROL NEEDS THE $WS_CLIPSIBLINGS STYLE!
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _ISN_InputAddClearButton($InputHandle = "", $ResizeValue = 0, $ClickFunc = "", $BackgroundOverwrite = -1)
	;NOTE: INPUT CONTROL NEEDS THE $WS_CLIPSIBLINGS STYLE!
	Local $OwnerWindow = _WinAPI_GetAncestor(GUICtrlGetHandle($InputHandle), $GA_PARENT)
	Local $Controlpos = _ControlGetPos($OwnerWindow, "", $InputHandle)
	Local $IconMargin = 4 * $DPI
	Local $IconHeight = 16 * $DPI
	Local $IconWidth = 16 * $DPI
	If Not IsArray($Controlpos) Then Return 0
	GUISwitch($OwnerWindow)
	_GUICtrlEdit_SetMargins($InputHandle, BitOR($EC_LEFTMARGIN, $EC_RIGHTMARGIN), 0, $Controlpos[3])
	If $Skin_is_used = "true" Then $BackgroundOverwrite = $Skriptbaum_Suchfeld_Hintergrundfarbe
	If $BackgroundOverwrite = -1 Then
		Local $NewButton = GUICtrlCreatePic("", $Controlpos[0] + ($Controlpos[2] - $IconWidth - $IconMargin), $Controlpos[1] + ($Controlpos[3] / 2) - ($IconHeight / 2), $IconWidth, $IconHeight)
;~ 		Local $NewButton = GUICtrlCreatePic("", $Controlpos[0] + ($Controlpos[2] - (16 * $DPI) - $IconMargin), $Controlpos[1] + ($Controlpos[3] / 2) - (16 / 2), 16, 16)
		_SetIconAlpha($NewButton, $smallIconsdll, 1937, 16, 16)
	Else
		Local $NewButton = GUICtrlCreateIcon("", 0, $Controlpos[0] + ($Controlpos[2] - $IconWidth - $IconMargin), $Controlpos[1] + ($Controlpos[3] / 2) - ($IconHeight / 2), $IconWidth, $IconHeight)
;~ 		Local $NewButton = GUICtrlCreateIcon("", 0, $Controlpos[0] + ($Controlpos[2] - (16 * $DPI) - $IconMargin), $Controlpos[1] + ($Controlpos[3] / 2) - (16 / 2), 16, 16)
		_SetCombineBkIcon($NewButton, $BackgroundOverwrite, $smallIconsdll, 1937 - 1, $IconWidth, $IconHeight, "", 0, -1, -1, 0, 0)
	EndIf
	GUICtrlSetCursor($NewButton, 0)
	GUICtrlSetOnEvent($NewButton, $ClickFunc)
	GUICtrlSetResizing($NewButton, $ResizeValue)
	GUICtrlSetState($NewButton, $GUI_ONTOP)
	Return $NewButton
EndFunc   ;==>_ISN_InputAddClearButton

; #FUNCTION# ====================================================================================================================
; Name ..........: _ISN_ReposClearButton_in_Input
; Description ...: Re position the clear button in an input control
; Syntax ........: _ISN_ReposClearButton_in_Input([$InputHandle = "" [, $ClearButtonHandle = ""]])
; Parameters ....: $InputHandle         - [optional] Handle to the input control.
;                  $ClearButtonHandle   - [optional] Handle to the ClearButton, returned by "_ISN_InputAddClearButton"
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _ISN_ReposClearButton_in_Input($InputHandle = "", $ClearButtonHandle = "")
	Local $OwnerWindow = _WinAPI_GetAncestor(GUICtrlGetHandle($InputHandle), $GA_PARENT)
	Local $Controlpos = _ControlGetPos($OwnerWindow, "", $InputHandle)
	Local $IconMargin = 4 * $DPI
	Local $IconHeight = 16 * $DPI
	Local $IconWidth = 16 * $DPI
	If Not IsArray($Controlpos) Then Return 0
	Return GUICtrlSetPos($ClearButtonHandle, $Controlpos[0] + ($Controlpos[2] - $IconWidth - $IconMargin), $Controlpos[1] + ($Controlpos[3] / 2) - ($IconHeight / 2), $IconWidth, $IconHeight)
;~ 	Return GUICtrlSetPos($ClearButtonHandle, $Controlpos[0] + ($Controlpos[2] - (16 * $DPI) - $IconMargin), $Controlpos[1] + ($Controlpos[3] / 2) - (16 / 2), 16, 16)
;~ 	return GUICtrlSetPos($ClearButtonHandle,$Controlpos[0]+($Controlpos[2]-(16 * $DPI)-$IconMargin),$Controlpos[1]+($Controlpos[3]/2)-8,16,16)
EndFunc   ;==>_ISN_ReposClearButton_in_Input


Func _ISN_ReturnPathToAutoItEXE()
	If $ISN_AutoitArchitecture = "x86" Then
		Return $autoitexe
	Else
		Return $autoitx64exe
	EndIf
EndFunc   ;==>_ISN_ReturnPathToAutoItEXE


; #FUNCTION# =========================================================================================================
; Name...........: GUICtrlGetBkColor
; Description ...: Retrieves the RGB value of the control background.
; Syntax.........: GUICtrlGetBkColor($iControlID)
; Parameters ....: $iControlID - A valid control ID.
; Requirement(s).: v3.3.2.0 or higher
; Return values .: Success - Returns RGB value of the control background.
;                  Failure - Returns 0 & sets @error = 1
; Author ........: guinness & additional information from Yashied for WinAPIEx.
; Example........; Yes
;=====================================================================================================================

Func GUICtrlGetBkColor($iControlID)
	Local $bGetBkColor, $hDC, $hHandle
	$hHandle = GUICtrlGetHandle($iControlID)
	$hDC = _WinAPI_GetDC($hHandle)
	$bGetBkColor = _WinAPI_GetPixel($hDC, 2, 2)
	_WinAPI_ReleaseDC($hHandle, $hDC)
	Return $bGetBkColor
EndFunc   ;==>GUICtrlGetBkColor

; #FUNCTION# ====================================================================================================================
; Name ..........: _Input_Error_FX
; Description ...: Flashes the background color of an input control red, an slowly fade back to the normal color.
; Syntax ........: _Input_Error_FX([$Control = ""])
; Parameters ....: $Control             - [optional] An unknown value. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _Input_Error_FX($Control = "")
	;by ISI360
	If $Control = "" Then Return
	If $Control_Flashes = 1 Then Return
	$Control_Flashes = 1
	$old_bg = "0x" & Hex(GUICtrlGetBkColor($Control), 6)
	$old_red = _ColorGetRed($old_bg)
	$old_green = _ColorGetGreen($old_bg)
	$old_blue = _ColorGetBlue($old_bg)
	GUICtrlSetBkColor($Control, 0xFB6969)
	Sleep(100)
	$new_bg = "0x" & Hex(GUICtrlGetBkColor($Control), 6)
	$new_red = _ColorGetRed($new_bg)
	$new_green = _ColorGetGreen($new_bg)
	$new_blue = _ColorGetBlue($new_bg)
	$steps = 5
	Sleep(300)
	While 1
		$new_red = $new_red - $steps
		If $new_red < $old_red Then $new_red = $old_red
		$new_green = $new_green + $steps
		If $new_green > $old_green Then $new_green = $old_green
		$new_blue = $new_blue + $steps
		If $new_blue > $old_blue Then $new_blue = $old_blue
		$bg = "0x" & Hex($new_red, 2) & Hex($new_green, 2) & Hex($new_blue, 2)
		If $new_red = $old_red And $new_green = $old_green And $new_blue = $old_blue Then ExitLoop
		GUICtrlSetBkColor($Control, $bg)
		Sleep(20)
	WEnd
	GUICtrlSetBkColor($Control, $old_bg)
	$Control_Flashes = 0
EndFunc   ;==>_Input_Error_FX

;==================================================================================================
; Function Name:   _GUICtrlTreeView_ExpandOneLevel($hTreeView [, $hParentItem=0])
; Description::    Ausklappen nur EINER Ebene eines Items, analog zum Mausklick auf '+'
; Parameter(s):    $hTreeView     Handle des TreeView
;                  $hParentItem   Handle des Auszuklappenden Parent-Items
;                                 Standard 0 ==> Handle des ersten Item im TreeView
; Return:          Erfolg         nichts
;                  Fehler         @error 1  -  TreeView enthlt kein Item
;                                 @error 2  -  Item hat keine Child-Item
; Note:            Die Funktion sollte zwischen _GUICtrlTreeView_BeginUpdate() und _GUICtrlTreeView_EndUpdate()
;                  ausgefhrt werden um ein Flackern zu verhindern
; Author(s):       BugFix (bugfix@autoit.de)
;==================================================================================================

Func _GUICtrlTreeView_ExpandOneLevel($hTreeView, $hParentItem = 0)
	If $hParentItem < 1 Then
		Local $hCurrentItem = _GUICtrlTreeView_GetFirstItem($hTreeView)
	Else
		Local $hCurrentItem = $hParentItem
	EndIf
	If $hCurrentItem = 0 Then Return SetError(1)
	Local $hChild
	Local $countChild = _GUICtrlTreeView_GetChildCount($hTreeView, $hCurrentItem)
	If $countChild = 0 Then Return SetError(2)
	_GUICtrlTreeView_Expand($hTreeView, $hCurrentItem)
	For $i = 1 To $countChild
		If $i = 1 Then
			$hChild = _GUICtrlTreeView_GetFirstChild($hTreeView, $hCurrentItem)
		Else
			$hChild = _GUICtrlTreeView_GetNextSibling($hTreeView, $hChild)
		EndIf
		If _GUICtrlTreeView_GetChildren($hTreeView, $hChild) Then _GUICtrlTreeView_Expand($hTreeView, $hChild, False)
	Next
EndFunc   ;==>_GUICtrlTreeView_ExpandOneLevel

; #FUNCTION# ====================================================================================================================
; Name...........: _ReduceMemory
; Author ........: w_Outer, Rajesh V R, Prog@ndy
; ===============================================================================================================================

Func _ReduceMemory($iPid = -1)
	If $iPid = -1 Or ProcessExists($iPid) = 0 Then
		Local $ai_GetCurrentProcess = DllCall('kernel32.dll', 'ptr', 'GetCurrentProcess')
		Local $ai_Return = DllCall("psapi.dll", 'int', 'EmptyWorkingSet', 'ptr', $ai_GetCurrentProcess[0])
		Return $ai_Return[0]
	EndIf

	Local $ai_Handle = DllCall("kernel32.dll", 'ptr', 'OpenProcess', 'int', 0x1F0FFF, 'int', False, 'int', $iPid)
	Local $ai_Return = DllCall("psapi.dll", 'int', 'EmptyWorkingSet', 'ptr', $ai_Handle[0])
	DllCall('kernel32.dll', 'int', 'CloseHandle', 'ptr', $ai_Handle[0])
	Return $ai_Return[0]
EndFunc   ;==>_ReduceMemory


; #FUNCTION# ====================================================================================================================
; Name ..........: _ist_windows_vista_oder_hoeher
; Description ...: Checks, if the used OS is Win. vista or higher
; Syntax ........: _ist_windows_vista_oder_hoeher()
; Parameters ....: None
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _ist_windows_vista_oder_hoeher()
	Switch @OSVersion
		Case "WIN_2000", "WIN_2003", "WIN_XP", "WIN_XPe"
			Return 0
	EndSwitch
	Return 1
EndFunc   ;==>_ist_windows_vista_oder_hoeher

; #FUNCTION# ====================================================================================================================
; Name ..........: _ist_windows_8_oder_hoeher
; Description ...: Checks, if the used OS is Win. 8 or higher
; Syntax ........: _ist_windows_8_oder_hoeher()
; Parameters ....: None
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _ist_windows_8_oder_hoeher()
	Switch @OSVersion
		Case "WIN_2000", "WIN_2003", "WIN_XP", "WIN_XPe", "WIN_VISTA", "WIN_7", "WIN_2003", "WIN_2008"
			Return 0

	EndSwitch
	Return 1
EndFunc   ;==>_ist_windows_8_oder_hoeher


; #FUNCTION# ====================================================================================================================
; Name ..........: _DesktopDimensions
; Description ...: Returns an array containing information about the primary and virtual monitors.
; Syntax ........: _DesktopDimensions()
; Return values .: Success - Returns a 6-element array containing the following information:
;                  $aArray[0] = Total number of monitors.
;                  $aArray[1] = Width of the primary monitor.
;                  $aArray[2] = Height of the primary monitor.
;                  $aArray[3] = Total width of the desktop including the width of multiple monitors. Note: If no secondary monitor this will be the same as $aArray[2].
;                  $aArray[4] = Total height of the desktop including the height of multiple monitors. Note: If no secondary monitor this will be the same as $aArray[3].
; Author ........: guinness
; Remarks .......: WinAPI.au3 must be included i.e. #include <WinAPI.au3>
; Related .......: @DesktopWidth, @DesktopHeight, _WinAPI_GetSystemMetrics
; Example .......: Yes
; ===============================================================================================================================
Func _DesktopDimensions()
	Local $aReturn = [_WinAPI_GetSystemMetrics($SM_CMONITORS), _ ; Number of monitors.
			_WinAPI_GetSystemMetrics($SM_CXSCREEN), _ ; Width or Primary monitor.
			_WinAPI_GetSystemMetrics($SM_CYSCREEN), _ ; Height or Primary monitor.
			_WinAPI_GetSystemMetrics($SM_CXVIRTUALSCREEN), _ ; Width of the Virtual screen.
			_WinAPI_GetSystemMetrics($SM_CYVIRTUALSCREEN)] ; Height of the Virtual screen.
	Return $aReturn
EndFunc   ;==>_DesktopDimensions


;==================================================================================================
; Function Name:   _MonitorEnumProc($hMonitor, $hDC, $lRect, $lParam)
; Description::    Enum Callback Function for EnumDisplayMonitors in _GetMonitors
; Author(s):       xrxca (autoit@forums.xrx.ca)
;==================================================================================================

Func _MonitorEnumProc($hMonitor, $hDC, $lRect, $lParam)
	Local $Rect = DllStructCreate("int left;int top;int right;int bottom", $lRect)
	$__MonitorList[0][0] += 1
	ReDim $__MonitorList[$__MonitorList[0][0] + 1][5]
	$__MonitorList[$__MonitorList[0][0]][0] = $hMonitor
	$__MonitorList[$__MonitorList[0][0]][1] = DllStructGetData($Rect, "left")
	$__MonitorList[$__MonitorList[0][0]][2] = DllStructGetData($Rect, "top")
	$__MonitorList[$__MonitorList[0][0]][3] = DllStructGetData($Rect, "right")
	$__MonitorList[$__MonitorList[0][0]][4] = DllStructGetData($Rect, "bottom")
	Return 1 ; Return 1 to continue enumeration
EndFunc   ;==>_MonitorEnumProc

;============================================================================================== _NumberAndNameMonitors
; Function Name:    _NumberAndNameMonitors ()
; Description:   Provides the first key elements of a multimonitor system, included the Regedit Keys
; Parameter(s):   None
; Return Value(s):   $NumberAndName [][]
;~        [0][0] total number of video devices
;;       [x][1] name of the device
;;       [x][2] name of the adapter
;;       [x][3] monitor flags (value is returned in Hex str -convert in DEC before use with Bitand)
;;       [x][4] registry key of the device
; Remarks:   the flag value [x][3] can be one of the following
;;       DISPLAY_DEVICE_ATTACHED_TO_DESKTOP  0x00000001
;;             DISPLAY_DEVICE_MULTI_DRIVER       0x00000002
;;            DISPLAY_DEVICE_PRIMARY_DEVICE    0x00000004
;;            DISPLAY_DEVICE_VGA               0x00000010
;;        DISPLAY_MIRROR_DEVICE  0X00000008
;;        DISPLAY_REMOVABLE  0X00000020
;
; Author(s):        Hermano
;===========================================================================================================================
Func _get_primary_monitor()
	Local $aScreenResolution = _DesktopDimensions()
	If Not IsArray($aScreenResolution) Then Return 1
	Return _GetMonitorFromPoint(($aScreenResolution[1] / 2), ($aScreenResolution[2] / 2))

	;Funktioniert ab Win10 nicht mehr zu 100%
;~ 	Local $dev = -1, $id = 0, $msg_ = "", $EnumDisplays, $StateFlag
;~ 	Dim $NumberAndName[2][6]
;~ 	Local $DISPLAY_DEVICE = DllStructCreate("int;char[32];char[128];int;char[128];char[128]")
;~ 	DllStructSetData($DISPLAY_DEVICE, 1, DllStructGetSize($DISPLAY_DEVICE))
;~ 	Dim $dll = "user32.dll"
;~ 	Do
;~ 		$dev += 1
;~ 		$EnumDisplays = DllCall($dll, "int", "EnumDisplayDevices", "ptr", 0, "int", $dev, "ptr", DllStructGetPtr($DISPLAY_DEVICE), "int", 1)
;~ 		If $EnumDisplays[0] <> 0 Then
;~ 			ReDim $NumberAndName[$dev + 2][6]
;~ 			$NumberAndName[$dev + 1][1] = DllStructGetData($DISPLAY_DEVICE, 2) ;device Name
;~ 			$NumberAndName[$dev + 1][2] = DllStructGetData($DISPLAY_DEVICE, 3) ;device or display description
;~ 			$NumberAndName[$dev + 1][3] = Hex(DllStructGetData($DISPLAY_DEVICE, 4)) ;all flags (value in HEX)
;~ 			$NumberAndName[$dev + 1][4] = DllStructGetData($DISPLAY_DEVICE, 6) ;registry key of the device
;~ 			$NumberAndName[$dev + 1][5] = DllStructGetData($DISPLAY_DEVICE, 5) ;hardware interface name
;~ 		EndIf
;~ 	Until $EnumDisplays[0] = 0
;~ 	$NumberAndName[0][0] += $dev
;~ 	For $x = 0 To $NumberAndName[0][0]
;~ 		If BitAND($NumberAndName[$x][3], 0x00000004) Then
;~ 			Return $x
;~ 		EndIf
;~ 	Next
;~ 	Return 1
EndFunc   ;==>_get_primary_monitor

;==================================================================================================
; Function Name:   _GetMonitors()
; Description::    Load monitor positions
; Parameter(s):    n/a
; Return Value(s): 2D Array of Monitors
;                       [0][0] = Number of Monitors
;                       [i][0] = HMONITOR handle of this monitor.
;                       [i][1] = Left Position of Monitor
;                       [i][2] = Top Position of Monitor
;                       [i][3] = Right Position of Monitor
;                       [i][4] = Bottom Position of Monitor
; Note:            [0][1..4] are set to Left,Top,Right,Bottom of entire screen
;                  hMonitor is returned in [i][0], but no longer used by these routines.
;                  Also sets $__MonitorList global variable (for other subs to use)
; Author(s):       xrxca (autoit@forums.xrx.ca)
;==================================================================================================

Func _GetMonitors()
	$__MonitorList[0][0] = 0 ;  Added so that the global array is reset if this is called multiple times
	Local $handle = DllCallbackRegister("_MonitorEnumProc", "int", "hwnd;hwnd;ptr;lparam")
	DllCall("user32.dll", "int", "EnumDisplayMonitors", "hwnd", 0, "ptr", 0, "ptr", DllCallbackGetPtr($handle), "lparam", 0)
	DllCallbackFree($handle)
	Local $i = 0
	For $i = 1 To $__MonitorList[0][0]
		If $__MonitorList[$i][1] < $__MonitorList[0][1] Then $__MonitorList[0][1] = $__MonitorList[$i][1]
		If $__MonitorList[$i][2] < $__MonitorList[0][2] Then $__MonitorList[0][2] = $__MonitorList[$i][2]
		If $__MonitorList[$i][3] > $__MonitorList[0][3] Then $__MonitorList[0][3] = $__MonitorList[$i][3]
		If $__MonitorList[$i][4] > $__MonitorList[0][4] Then $__MonitorList[0][4] = $__MonitorList[$i][4]
	Next
	Return $__MonitorList
EndFunc   ;==>_GetMonitors

;==================================================================================================
; Function Name:   _GetMonitorFromPoint([$XorPoint = -654321[, $Y = 0]])
; Description::    Get a monitor number from an x/y pos or the current mouse position
; Parameter(s):
;     optional:    $XorPoint X Position or Array with X/Y as items 0,1 (ie from MouseGetPos())
;     optional:    $Y        Y Position
; Note:            Should probably have specified return/error codes but haven't put them in yet,
;                  and better checking should be done on passed variables.
;                  Used to use MonitorFromPoint DLL call, but it didn't seem to always work.
; Author(s):       xrxca (autoit@forums.xrx.ca)
;==================================================================================================

Func _GetMonitorFromPoint($XorPoint = 0, $y = 0)
	If @NumParams = 0 Then
		Local $MousePos = MouseGetPos()
		Local $myX = $MousePos[0]
		Local $myY = $MousePos[1]
	ElseIf (@NumParams = 1) And IsArray($XorPoint) Then
		Local $myX = $XorPoint[0]
		Local $myY = $XorPoint[1]
	Else
		Local $myX = $XorPoint
		Local $myY = $y
	EndIf
	If $__MonitorList[0][0] == 0 Then
		_GetMonitors()
	EndIf
	Local $i = 0
	Local $Monitor = 0
	For $i = 1 To $__MonitorList[0][0]
		If ($myX >= $__MonitorList[$i][1]) _
				And ($myX < $__MonitorList[$i][3]) _
				And ($myY >= $__MonitorList[$i][2]) _
				And ($myY < $__MonitorList[$i][4]) Then $Monitor = $i
	Next
	Return $Monitor
EndFunc   ;==>_GetMonitorFromPoint

;==================================================================================================
; Function Name:   _ShowMonitorInfo()
; Description::    Show the info in $__MonitorList in a msgbox (line 0 is entire screen)
; Parameter(s):    n/a
; Return Value(s): n/a
; Author(s):       xrxca (autoit@forums.xrx.ca)
;==================================================================================================

Func _ShowMonitorInfo()
	If $__MonitorList[0][0] == 0 Then
		_GetMonitors()
	EndIf
	Local $msg = ""
	Local $i = 0
	For $i = 0 To $__MonitorList[0][0]
		$msg &= $i & " - L:" & $__MonitorList[$i][1] & ", T:" & $__MonitorList[$i][2]
		$msg &= ", R:" & $__MonitorList[$i][3] & ", B:" & $__MonitorList[$i][4]
		If $i < $__MonitorList[0][0] Then $msg &= @CRLF
	Next
	MsgBox(0, $__MonitorList[0][0] & " Monitors: ", $msg)
EndFunc   ;==>_ShowMonitorInfo

;==================================================================================================
; Function Name:   _MaxOnMonitor($Title[, $Text = ''[, $Monitor = -1]])
; Description::    Maximize a window on a specific monitor (or the monitor the mouse is on)
; Parameter(s):    $Title   The title of the window to Move/Maximize
;     optional:    $Text    The text of the window to Move/Maximize
;     optional:    $Monitor The monitor to move to (1..NumMonitors) defaults to monitor mouse is on
; Note:            Should probably have specified return/error codes but haven't put them in yet
; Author(s):       xrxca (autoit@forums.xrx.ca)
;==================================================================================================

Func _MaxOnMonitor($Title, $text = '', $Monitor = -1)
	_CenterOnMonitor($Title, $text, $Monitor)
	WinSetState($Title, $text, @SW_MAXIMIZE)
EndFunc   ;==>_MaxOnMonitor

;==================================================================================================
; Function Name:   _CenterOnMonitor($Title[, $Text = ''[, $Monitor = -1]])
; Description::    Center a window on a specific monitor (or the monitor the mouse is on)
; Parameter(s):    $Title   The title of the window to Move/Maximize
;     optional:    $Text    The text of the window to Move/Maximize
;     optional:    $Monitor The monitor to move to (1..NumMonitors) defaults to monitor mouse is on
;					$Ignore_primary Ist nur 1 wenn Monitore Identifiziert werden (dadurch wird _get_primary_monitor() bersprungen)
; Note:            Should probably have specified return/error codes but haven't put them in yet
; Author(s):       xrxca (autoit@forums.xrx.ca)
;==================================================================================================

Func _CenterOnMonitor($Title, $text = '', $Monitor = -1, $Ignore_primary = 0)
	If $Monitor = -1 Then $Monitor = $Runonmonitor
	If $Immer_am_primaeren_monitor_starten = "true" And $Ignore_primary = 0 Then $Monitor = _get_primary_monitor()
	$hWindow = WinGetHandle($Title, $text)
	If $Monitor > $__MonitorList[0][0] Then $Monitor = 1
	If Not @error Then
		If $Monitor == -1 Then
			$Monitor = _GetMonitorFromPoint()
		ElseIf $__MonitorList[0][0] == 0 Then
			_GetMonitors()
		EndIf
		If ($Monitor > 0) And ($Monitor <= $__MonitorList[0][0]) Then
			; Restore the window if necessary
			Local $WinState = WinGetState($hWindow)
			If BitAND($WinState, 16) Or BitAND($WinState, 32) Then
				WinSetState($hWindow, '', @SW_RESTORE)
			EndIf
			Local $WinSize = _WinGetPos($hWindow)
			Local $x = Int(($__MonitorList[$Monitor][3] - $__MonitorList[$Monitor][1] - $WinSize[2]) / 2) + $__MonitorList[$Monitor][1]
			Local $y = Int(($__MonitorList[$Monitor][4] - $__MonitorList[$Monitor][2] - $WinSize[3]) / 2) + $__MonitorList[$Monitor][2]
			WinMove($hWindow, '', $x, $y)
		EndIf
	EndIf
EndFunc   ;==>_CenterOnMonitor



; #FUNCTION# ====================================================================================================================
; Name ..........: _Get_Monitor_Resolution
; Description ...:
; Syntax ........: _Get_Monitor_Resolution([$Monitor = -1])
; Parameters ....: $Monitor             - [optional] An unknown value. Default is -1.
; Return values .: None
; Author ........:
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _Get_Monitor_Resolution($Monitor = -1)
	Local $array[2]
	If $Immer_am_primaeren_monitor_starten = "true" Then $Monitor = _get_primary_monitor()
	If $Monitor = -1 Then Return
	If $Monitor > $__MonitorList[0][0] Then $Monitor = 1
	If Not @error Then
		If $Monitor == -1 Then
			$Monitor = _GetMonitorFromPoint()
		ElseIf $__MonitorList[0][0] == 0 Then
			_GetMonitors()
		EndIf
		If ($Monitor > 0) And ($Monitor <= $__MonitorList[0][0]) Then
			Local $width = Int(($__MonitorList[$Monitor][3] - $__MonitorList[$Monitor][1]) / 2) * 2
			Local $height = Int(($__MonitorList[$Monitor][4] - $__MonitorList[$Monitor][2])) + $__MonitorList[$Monitor][2]
			$array[0] = $width
			$array[1] = $height
			Return $array
		EndIf
	EndIf
EndFunc   ;==>_Get_Monitor_Resolution


; #FUNCTION# ====================================================================================================================
; Name ..........: _Monitor_Get_Resolution
; Description ...:
; Syntax ........: _Monitor_Get_Resolution([$Monitor = -1])
; Parameters ....: $Monitor             - [optional] An unknown value. Default is -1.
; Return values .: None
; Author ........:
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _Monitor_Get_Resolution($Monitor = -1)
	Local $array[4]
	If $Monitor = -1 Then Return
	If $Monitor > $__MonitorList[0][0] Then $Monitor = 1
	If Not @error Then
		If $Monitor == -1 Then
			$Monitor = _GetMonitorFromPoint()
		ElseIf $__MonitorList[0][0] == 0 Then
			_GetMonitors()
		EndIf
		If ($Monitor > 0) And ($Monitor <= $__MonitorList[0][0]) Then
			Local $width = Int(($__MonitorList[$Monitor][3] - $__MonitorList[$Monitor][1]) / 2) * 2
			Local $height = Int(($__MonitorList[$Monitor][4] - $__MonitorList[$Monitor][2])) + $__MonitorList[$Monitor][2]
			Local $x = Int(($__MonitorList[$Monitor][1]))
			Local $y = Int(($__MonitorList[$Monitor][2]))
			$array[0] = $x
			$array[1] = $y
			$array[2] = $width
			$array[3] = $height
			Return $array
		EndIf
	EndIf
EndFunc   ;==>_Monitor_Get_Resolution


; #FUNCTION# ====================================================================================================================
; Name ..........: SetBitmap
; Description ...:
; Syntax ........: SetBitmap($hGUI, $hImage, $iOpacity)
; Parameters ....: $hGUI                - A handle value.
;                  $hImage              - A handle value.
;                  $iOpacity            - A integer value.
; Return values .: None
; Author ........:
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func SetBitmap($hGUI, $hImage, $iOpacity)
	Local $hScrDC, $hMemDC, $hBitmap, $hOld, $pSize, $tSize, $pSource, $tSource, $pBlend, $tBlend

	$hScrDC = _WinAPI_GetDC(0)
	$hMemDC = _WinAPI_CreateCompatibleDC($hScrDC)
	$hBitmap = _GDIPlus_BitmapCreateHBITMAPFromBitmap($hImage)
	$hOld = _WinAPI_SelectObject($hMemDC, $hBitmap)
	$tSize = DllStructCreate($tagSIZE)
	$pSize = DllStructGetPtr($tSize)

	DllStructSetData($tSize, "X", _GDIPlus_ImageGetWidth($hImage))
	DllStructSetData($tSize, "Y", _GDIPlus_ImageGetHeight($hImage))
	$tSource = DllStructCreate($tagPOINT)
	$pSource = DllStructGetPtr($tSource)
	$tBlend = DllStructCreate($tagBLENDFUNCTION)
	$pBlend = DllStructGetPtr($tBlend)
	DllStructSetData($tBlend, "Alpha", $iOpacity)
	DllStructSetData($tBlend, "Format", $AC_SRC_ALPHA)
	_WinAPI_UpdateLayeredWindow($hGUI, $hScrDC, 0, $pSize, $hMemDC, $pSource, 0, $pBlend, $ULW_ALPHA)
	_WinAPI_ReleaseDC(0, $hScrDC)
	_WinAPI_SelectObject($hMemDC, $hOld)
	_WinAPI_DeleteObject($hBitmap)
	_WinAPI_DeleteDC($hMemDC)
EndFunc   ;==>SetBitmap

; #FUNCTION# ====================================================================================================================
; Name ..........: _SetIconAlpha
; Description ...: Draws an transparent Icon to an Picture control.
; Syntax ........: _SetIconAlpha($hWnd, $sIcon, $iIndex, $iWidth, $iHeight)
; Parameters ....: $hWnd                - A handle value.
;                  $sIcon               - A string value.
;                  $iIndex              - A integer value.
;                  $iWidth              - An AutoIt controlID.
;                  $iHeight             - A integer value.
; Return values .: None
; Author ........:
; Modified ......:
; Remarks .......: Used in the "_ISN_InputAddClearButton" func
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _SetIconAlpha($hWnd, $sIcon, $iIndex, $iWidth, $iHeight)

	If Not IsHWnd($hWnd) Then
		$hWnd = GUICtrlGetHandle($hWnd)
		If $hWnd = 0 Then
			Return SetError(1, 0, 0)
		EndIf
	EndIf

	$iWidth = $iWidth * $DPI
	$iHeight = $iHeight * $DPI

	If $iIndex <> 0 Then $iIndex = $iIndex - 1
	Local $hIcon = _WinAPI_ShellExtractIcon($sIcon, $iIndex, $iWidth, $iHeight)

	If $hIcon = 0 Then
		Return SetError(1, 0, 0)
	EndIf

	Local $hBitmap, $hObj, $hDC, $hMem, $hSv

	$hDC = _WinAPI_GetDC($hWnd)
	$hMem = _WinAPI_CreateCompatibleDC($hDC)
	$hBitmap = _WinAPI_CreateCompatibleBitmap($hDC, $iWidth, $iHeight)
	$hSv = _WinAPI_SelectObject($hMem, $hBitmap)
	_WinAPI_DrawIconEx($hMem, 0, 0, $hIcon, $iWidth, $iHeight, 0, 0, 2)
	_WinAPI_ReleaseDC($hWnd, $hDC)
	_WinAPI_SelectObject($hMem, $hSv)
	_WinAPI_DeleteDC($hMem)
	_WinAPI_DestroyIcon($hIcon)
	_WinAPI_DeleteObject(_SendMessage($hWnd, 0x0172, 0, 0))
	_SendMessage($hWnd, 0x0172, 0, $hBitmap)
	$hObj = _SendMessage($hWnd, 0x0173)
	If $hObj <> $hBitmap Then
		_WinAPI_DeleteObject($hBitmap)
	EndIf
	Return 1
EndFunc   ;==>_SetIconAlpha



; #FUNCTION# ====================================================================================================================
; Name ..........: Button_AddIcon
; Description ...: Adds an icon to a button control.
; Syntax ........: Button_AddIcon($nID, $sIconFile, $nIconID, $nAlign)
; Parameters ....: $nID                 - Handle to the Button control
;                  $sIconFile           - Icon file (or dll) to us
;                  $nIconID             - Icon Index to use
;                  $nAlign              - 0 = left; 1 = right; 2 = top; 3 = bottom; 4 = center
; Return values .: None
; Author ........:
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func Button_AddIcon($nID, $sIconFile, $nIconID, $nAlign)
;~ 	$sIconFile = $smallIconsdll
	If $DPI > 1.99 Then
		Local $hIL = ImageList_Create(24, 24, BitOR($ILC_MASK, $ILC_COLOR32), 0, 1) ;Use bigger icons over 200 % dpi scaling
	Else
		Local $hIL = ImageList_Create(16, 16, BitOR($ILC_MASK, $ILC_COLOR32), 0, 1)
	EndIf
	Local $stIcon = DllStructCreate("int")
	ExtractIconEx($sIconFile, $nIconID, DllStructGetPtr($stIcon), 0, 1)
	ImageList_AddIcon($hIL, DllStructGetData($stIcon, 1))
	DestroyIcon(DllStructGetData($stIcon, 1))
	Local $stBIL = DllStructCreate("dword;int[4];uint")
	DllStructSetData($stBIL, 1, $hIL)
	DllStructSetData($stBIL, 2, 1, 1)
	DllStructSetData($stBIL, 2, 1, 2)
	DllStructSetData($stBIL, 2, 1, 3)
	DllStructSetData($stBIL, 2, 1, 4)
	DllStructSetData($stBIL, 3, $nAlign)

	Local $LeftOffset = 1
	If $nAlign = 0 Then $LeftOffset = 7 * $DPI
	_GUICtrlButton_SetImageList($nID, $hIL, $nAlign, $LeftOffset, 1, 1, 1)

	;GUICtrlSendMsg($nID, $BCM_SETIMAGELIST, 0, DllStructGetPtr($stBIL))
EndFunc   ;==>Button_AddIcon

; #FUNCTION# ====================================================================================================================
; Name ..........: ExtractIconEx
; Description ...:
; Syntax ........: ExtractIconEx($sIconFile, $nIconID, $ptrIconLarge, $ptrIconSmall, $nIcons)
; Parameters ....: $sIconFile           - A string value.
;                  $nIconID             - An AutoIt controlID.
;                  $ptrIconLarge        - A pointer value.
;                  $ptrIconSmall        - A pointer value.
;                  $nIcons              - A floating point number value.
; Return values .: None
; Author ........:
; Modified ......:
; Remarks .......: Used with the "Button_AddIcon" func
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func ExtractIconEx($sIconFile, $nIconID, $ptrIconLarge, $ptrIconSmall, $nIcons)
	Local $nCount = DllCall('shell32.dll', 'int', 'ExtractIconEx', _
			'str', $sIconFile, _
			'int', $nIconID, _
			'ptr', $ptrIconLarge, _
			'ptr', $ptrIconSmall, _
			'int', $nIcons)
	Return $nCount[0]
EndFunc   ;==>ExtractIconEx


; #FUNCTION# ====================================================================================================================
; Name ..........: ArraySortUnique
; Description ...: Sorts an array and deletes double items.
; Syntax ........: ArraySortUnique(ByRef $avArray [, $iDescending = 0 [, $iStart = 0 [, $iEnd = 0 [, $iSubItem = 0]]]])
; Parameters ....: ByRef $avArray       - Array to sort
;                  $iDescending         - [optional] If set to 1, sort in descending order. Default is 0.
;                  $iStart              - [optional] Index of array to start sorting (default 0 = first element or row)
;                  $iEnd                - [optional] Index of array to stop sorting (default 0 = last element or row)
;                  $iSubItem            - [optional] Sub-index to sort on in 2D arrays (default 0 = first column)
; Return values .: None
; Author ........: Prog@ndy
; Modified ......:
; Remarks .......: If the equal entries are one after the other, delete them :)
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func ArraySortUnique(ByRef $avArray, $iDescending = 0, $iStart = 0, $iEnd = 0, $iSubItem = 0)
	If Not IsArray($avArray) Then Return
	Local $Ret = _ArraySort($avArray, $iDescending, $iStart, $iEnd, $iSubItem)
	If @error Then Return SetError(@error, 0, $Ret)
	Local $ResultIndex = 1, $ResultArray[UBound($avArray)]
	$ResultArray[0] = $avArray[0]
	For $i = 1 To UBound($avArray) - 1
		If $i > UBound($avArray) - 1 Then ExitLoop
		If Not ($avArray[$i] = $avArray[$i - 1]) Then
			$ResultArray[$ResultIndex] = $avArray[$i]
			$ResultIndex += 1
		EndIf
	Next
	ReDim $ResultArray[$ResultIndex]
	$avArray = $ResultArray
	Return 1
EndFunc   ;==>ArraySortUnique


; #FUNCTION# ====================================================================================================================
; Name ..........: _IsColorBright
; Description ...: Checks, if a RGB Hex color is a bright or a dark color.
; Syntax ........: _IsColorBright($_iColor)
; Parameters ....: $_iColor             - An unknown value.
; Return values .: True = Bright color; False = Dark color
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _IsColorBright($_iColor)
	Local $RGB = _ColorGetRGB($_iColor)
	For $i = 0 To 2
		If $RGB[$i] < 200 Then Return False
	Next
	Return True
EndFunc   ;==>_IsColorBright

;#FUNCTION#========================================================================================
; Name...........: _GuiSetDropShadow
; Description ...: Sets the drop shadow effect on forms and dialogs for current process
; Syntax.........: _GuiSetDropShadow($hwnd, $fDisrespectUser = True)
; Parameters ....: $hWnd               - Handle to parent form or child dialog (GuiCreate(), MsgBox(), FileOpenDialog(), etc.)
;                  $fDisrespectUser    - True: (Default) - set system option for drop shadow if disabled by user
;                                      - False:             - do not set system option for drop shadow if disabled by user
; Return values .: Success      - 1
;                  Failure      - 0       - @error set and @extended set to point of failure
; Author(s) ........: rover, (lod3n, Rasim for Get/SetClassLong, Kip - RegisterclassEx() for drop shadow idea, ProgAndy - xMsgBox hook)
; Remarks .......: Note: drop shadow is lost if parent form clicked on (If MsgBox created with parent handle)
;                                 hiding, then restoring MsgBox to foreground or moving MsgBox off of form restores drop shadow.
;                                 use 262144 or 4096 flags with MsgBox if using with hParent handle to prevent loss of drop shadow if parent clicked on.
;                                 this behaviour is apparently by design.
;+
;                  Minimum Operating Systems: Windows XP
; Related .......:
; Link ..........; @@MsdnLink@@ SetClassLong Function
; Example .......; Yes
; ===================================================================================================
Func _GuiSetDropShadow($hWnd, $fDisrespectUser = False)
	If Not IsHWnd($hWnd) Then Return SetError(1, 1, 0)

	;check if hWnd is from current process
	Local $aResult = DllCall("User32.dll", "int", "GetWindowThreadProcessId", "hwnd", $hWnd, "int*", 0)
	If @error Or $aResult[2] <> @AutoItPID Then Return SetError(@error, 2, 0)

	If Not IsDeclared("SPI_GETDROPSHADOW") Then Local Const $SPI_GETDROPSHADOW = 0x1024
	If Not IsDeclared("SPI_SETDROPSHADOW") Then Local Const $SPI_SETDROPSHADOW = 0x1025
	If Not IsDeclared("CS_DROPSHADOW") Then Local Const $CS_DROPSHADOW = 0x00020000
	If Not IsDeclared("GCL_STYLE") Then Local Const $GCL_STYLE = -26

	$aResult = DllCall("user32.dll", "int", "SystemParametersInfo", "int", $SPI_GETDROPSHADOW, "int", 0, "int*", 0, "int", 0)
	Local $iErr = @error
	If $iErr Or Not IsArray($aResult) Then Return SetError($iErr, 3, 0)

	;if 'Show shadows under menus' option not set, try activating it.
	If Not $aResult[3] And $fDisrespectUser Then
		;turn on drop shadows
		$aResult = DllCall("user32.dll", "int", "SystemParametersInfo", "int", $SPI_SETDROPSHADOW, "int", 0, "int", True, "int", 0)
		$iErr = @error
		If $iErr Or Not IsArray($aResult) Or $aResult[0] <> 1 Then Return SetError($iErr, 4, 0)
	EndIf

	;get styles from WndClassEx struct
	$aResult = DllCall("user32.dll", "dword", "GetClassLong", "hwnd", $hWnd, "int", $GCL_STYLE)
	$iErr = @error
	If $iErr Or Not IsArray($aResult) Or Not $aResult[0] Then Return SetError($iErr, 5, 0)
	Local $OldStyle = $aResult[0]

	;add drop shadow style to styles
	Local $Style = BitOR($OldStyle, $CS_DROPSHADOW)

	$aResult = DllCall("user32.dll", "dword", "SetClassLong", "hwnd", $hWnd, "int", $GCL_STYLE, "long", $Style)
	$iErr = @error
	If $iErr Or Not IsArray($aResult) Or Not $aResult[0] Then Return SetError($iErr, 7, 0)
	If $aResult[0] = $OldStyle Then Return SetError($iErr, 0, 1)
	Return SetError($iErr, 8, 0)

EndFunc   ;==>_GuiSetDropShadow

; #FUNCTION# ====================================================================================================================
; Name ..........: _WinGetPos
; Description ...: We use our own WinGetPos Func. The AutoIt built in func caused some inexplicable errors.
;				   (When a external process is parent of the ISN and has more of 300+ controls in a gui.)
; Syntax ........: _WinGetPos([$hGUI="" [, $Text=""]])
; Parameters ....: $hGUI                - [optional] A handle value. Default is "".
;                  $Text                - [optional] A dll struct value. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _WinGetPos($hGUI="",$Text="")
   Local $array[1]
   _ArrayDelete($array, 0)
   if not IsHWnd($hGUI) then $hGUI = GUICtrlGetHandle($hGUI)
   Local $tRECT = _WinAPI_GetWindowRect($hGUI)
   If @error Then Return SetError(@error, @extended, 0)
   _ArrayAdd($array,DllStructGetData($tRECT, "Left"))
   _ArrayAdd($array,DllStructGetData($tRECT, "Top"))
   _ArrayAdd($array,DllStructGetData($tRECT, "Right")-DllStructGetData($tRECT, "Left"))
   _ArrayAdd($array,DllStructGetData($tRECT, "Bottom")-DllStructGetData($tRECT, "Top"))
   return $array
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ControlGetPos
; Description ...: We use our own ControlGetPos Func. The AutoIt built in func caused some inexplicable errors.
;				   (When a external process is parent of the ISN and has more of 300+ controls in a gui.)
; Syntax ........: _ControlGetPos([$hGUI="" [, $Text="" [, $ControlID=""]]])
; Parameters ....: $hGUI                - [optional] A handle value. Default is "".
;                  $Text                - [optional] A dll struct value. Default is "".
;                  $ControlID           - [optional] An AutoIt controlID. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _ControlGetPos($hGUI="",$Text="",$ControlID="")
   if $hGUI = "" then $hGUI = _WinAPI_GetParent(GUICtrlGetHandle($ControlID))
   if $ControlID = "" then $ControlID = ControlGetHandle ($hGUI, "", "")
   Local $array[1]
   _ArrayDelete($array, 0)
   if not IsHWnd($ControlID) then $ControlID = GUICtrlGetHandle($ControlID)
   Local $tPoint = DllStructCreate("int X;int Y")
   Local $tRECT = _WinAPI_GetWindowRect($ControlID)
   If @error Then Return SetError(1)
   DllStructSetData($tPoint, "X", DllStructGetData($tRECT, "Left"))
   DllStructSetData($tPoint, "Y", DllStructGetData($tRECT, "Top"))
   _WinAPI_ScreenToClient($hGUI, $tPoint)
   _ArrayAdd($array,DllStructGetData($tPoint, "X"))
   _ArrayAdd($array,DllStructGetData($tPoint, "Y"))
   _ArrayAdd($array,DllStructGetData($tRECT, "Right")-DllStructGetData($tRECT, "Left"))
   _ArrayAdd($array,DllStructGetData($tRECT, "Bottom")-DllStructGetData($tRECT, "Top"))
   return $array
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _WinGetClientSize
; Description ...: We use our own WinGetClientSize Func. The AutoIt built in func caused some inexplicable errors.
;				   (When a external process is parent of the ISN and has more of 300+ controls in a gui.)
; Syntax ........: _WinGetClientSize([$hGUI="" [, $Text=""]])
; Parameters ....: $hGUI                - [optional] A handle value. Default is "".
;                  $Text                - [optional] A dll struct value. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......: No
; ===============================================================================================================================
Func _WinGetClientSize($hGUI="",$Text="")
   Local $array[1]
   _ArrayDelete($array, 0)
   Local $tRECT = _WinAPI_GetClientRect($hGUI)
   If @error Then Return SetError(@error, @extended, 0)
   _ArrayAdd($array,DllStructGetData($tRECT, "Right"))
   _ArrayAdd($array,DllStructGetData($tRECT, "Bottom"))
   return $array
EndFunc

; #FUNCTION# ====================================================================================================================
; Name ..........: _ISN_ReturnHandleWithDollar
; Description ...: 
; Syntax ........: _ISN_ReturnHandleWithDollar([$handle = ""])
; Parameters ....: $handle              - [optional] A handle value. Default is "".
; Return values .: None
; Author ........: ISI360
; Modified ......: 
; Remarks .......: 
; Related .......: 
; Link ..........: 
; Example .......: No
; ===============================================================================================================================
Func _ISN_ReturnHandleWithDollar($handle = "")
	If $handle = "" Then Return ""
	$handle = StringStripWS($handle, 3)
	$handle = StringReplace($handle, "$$", "")
	$handle = StringReplace($handle, "$$$", "")
	$handle = StringReplace($handle, "$$$$", "")
	If StringLeft($handle, 1) <> "$" Then
		Return "$" & $handle
	Else
		Return $handle
	EndIf
EndFunc 